/**********************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2004-2015 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
* =================================================================== *
* 																	  *
* THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY              *
* EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE *
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR  *
* PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL FREESCALE OR             *
* ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,    *
* SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT        *
* NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;        *
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)            *
* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, *
* STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)       *
* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED *
* OF THE POSSIBILITY OF SUCH DAMAGE.                                  *
* 																	  *
* =================================================================== */

/*****************************************************************************
* Simple Range Demo main file.
* 
*
* Authors: R44828, B50961, R20246
* Revision: 0.5
* Date: 24th of April 2015
*
* This project is for demo purpose only. It is not provided for development.
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
*
* The Simple Demonstration consists of two nodes:
*  TX Node
*  RX Node
* 
* Concerning the MRB-KW019032EU demo configuration 
* Three settings have to be chosen by the user in the file main.h:
*   - The RF frequency of the receiver (315MHz, 434MHz or 915MHz) 
*   - The way frames will be observed: 
*     with the Freescale LabVIEW Sensor GUI or with the hyperterminal.
*   - The MKW01 node and broadcast addresses
*
*****************************************************************************/

#include "main.h"
#include "ApplicationConf.h"                                                    /*Defines the Application default parameters*/
#include "SMAC.h"
#include "FunctionLib.h"
#include "Keyboard_interface.h"
#include "Uart_Interface.h"
#include "SMAC_Interface.h"
#include "Hyperterminal.h"
#include "LabVIEW_GUI.h"
#include "PhyISR.h" /* For CRC flag */


/************************************************************************************
*************************************************************************************
* Private prototypes 
*************************************************************************************
************************************************************************************/

void MCPSDataConfirm(txStatus_t TransmissionResult);
void MCPSDataIndication(rxPacket_t *gsRxPacket);

void TimerDelayCallback(tmrTimerID_t timerId);
void Keyboard_CallBack(uint8_t Keyboard_PortID);

void PlatformPortInit(void);
void InitGlobalVariables(void);
void Platform_Init(void);
void SetDefaultValues(void);
void InitSmac(void);
void DisplaySCI_MP(uint8_t *ptr_data,uint8_t len);
void DisplaySCI_MS(uint8_t *ptr_data,uint8_t len);

/************************************************************************************
*************************************************************************************
* Public memory declarations                                                                   
*************************************************************************************
************************************************************************************/
#define gRxNode_c 			     (2)
#define gTxNode_c                            (3)
 
#define gSimpleRangeDefaultPayloadLength_c   (27)

tmrTimerID_t delayTimer = gTmrInvalidTimerID_c;

static uint8_t gau8RxDataBuffer[gMaxSmacSDULength_c]; 
static uint8_t gau8TxDataBuffer[gMaxSmacSDULength_c]; 
 
static txPacket_t * gAppTxPacket;
static rxPacket_t * gAppRxPacket;

bool_t bTxDone;
bool_t bRxDone;
bool_t bDelayTimeout;
bool_t bTimeout;
bool_t bRxAborted; 


/************************************************************************************
*
* main
*
* Select the demo settings in the file main.h
*
************************************************************************************/
unsigned char tmpbuf[50];
uint16_t counter;
uint8_t Frame_Verification_Result;

void main(void) 
{
  DisableInterrupts();
  /* Initialization of MCU and transceiver  */
  Platform_Init();                                                              
  TurnOnLeds();                                                                 //All LEDs are lit to indicate the application has started.
  InitGlobalVariables();                                                        
  /* SMAC Initialization */
  InitSmac();
  /* Set Radio to default values */
  SetDefaultValues();      
  EnableInterrupts();

  Delay_ms(1000);                                                               //Wait 1 second to turn off Leds
  TurnOffLeds();                                                                

  #if (gRxNode_c == gDeviceType_c)
           (void)MLMERXEnableRequest(gAppRxPacket, 0);
  #endif
           
  #if (gTxNode_c == gDeviceType_c)
           (void)MCPSDataRequest(gAppTxPacket);
           TMR_StartTimer(delayTimer, gTmrSingleShotTimer_c, TmrMilliseconds(gDelayBetweenPacketsInMs_c), TimerDelayCallback);
  #endif 

  #if (FRAME_DISPLAY == HYPERTERMINAL)
           
  SendMessageToHyperterminal("FXTH87/MKW01 Demo",17);
  SendMessageToHyperterminal("Format of the frame:",20);
  SendMessageToHyperterminal("TireID Pressure AccelZ AccelX Voltage Temperature StatusAcq FrameVerificationResult Counter",91);

  #elif (FRAME_DISPLAY == FREESCALE_LABVIEW_SENSOR_GUI)
  
  Delay_ms(500);
  DisplayPopUp(" Freescale TPMS Demo",20);
  DisplayMessage("Alpha firmware TPMS Receiver MKW01 Rev 0.2",42);
  
  #endif // Frame display
  
  /* Counter initialisation => will be incremented each time a frame is sent to the UART */
  counter = 0x0000;
  
  for(;;) 
  {
    #if (gRxNode_c == gDeviceType_c)
                    
                     if(bRxDone)
                     {
                      
                         Led1Toggle();              //Toggle LED1 (D4 on TWR-RF board) upon packet Reception   
                         
                         Frame_Verification_Result = Frame_Verification();
                         
                         /*Upon a successful reception output the Rx buffer to the terminal*/                         
                                   
                         if ((FRAME_DISPLAY == HYPERTERMINAL) || (gau8RxDataBuffer[20] == 0x11))
                         {

                           SendDataToHyperterminal(tmpbuf, gau8RxDataBuffer, counter, Frame_Verification_Result);
                           
                           /*Line feed and carriage return*/
                           (void)UartX_Transmit("\n\r", 2, UartTxCallBack);
                           while (UartX_IsTxActive());
                           
                         }
                         else if ((FRAME_DISPLAY == FREESCALE_LABVIEW_SENSOR_GUI) || (gau8RxDataBuffer[20] == 0x22))
                         {
                           
                           SendDataToLabviewSensorGUI(tmpbuf, gau8RxDataBuffer, counter, Frame_Verification_Result);
                           
                         }
                         
                         /* Increment frame counter */
                         counter ++;           
                                                            
                         /* Enable a new Rx request */
                         (void)MLMERXEnableRequest(gAppRxPacket, 0);
                         bRxDone = FALSE;                         

                     }
                     
                     
                     if(bRxAborted){
                       
                      (void)MLMERXEnableRequest(gAppRxPacket, 0);
                       bRxAborted = FALSE;                      
                     }
    #endif
                     
                     
                                          
    #if (gTxNode_c == gDeviceType_c)

                     if(bDelayTimeout)
                     {
                             if(bTxDone)
                             {
                                     Led2Toggle();                              //Toggle LED2 (D3 on TWR-RF board) upon packet Reception
                                     (void)MCPSDataRequest(gAppTxPacket);
                                     TMR_StartTimer(delayTimer, gTmrSingleShotTimer_c, TmrMilliseconds(gDelayBetweenPacketsInMs_c), TimerDelayCallback);
                                     bDelayTimeout = FALSE;
                                     bTxDone = FALSE;
                             }
                             
                     }
    #endif 
	     
	   
  } 
}

 void SetDefaultValues(void)
   {
         (void) PhyPib_SetCarrierFreq (gDefaultRfFreq_c);                       //To choose in ApplicationConf.h
	 PhyPib_SetBitrate(gDefaultBitRate_c);                                  //50000 bps
	 PhyPib_SetFreqDeviation(gDefaultFreqDv_c);                             //25000 Hz                        
 	 (void)PhyPib_SetModulationType(gDefaultModulationType_c);              //FSK
 	 (void)PhyPib_SetModulationShaping(gDefaultModulationShaping_c);        //BT_1
	 (void)PhyPib_EnableSequencer(gDefaultSequencer_c);                     //ON
	 (void)PhyPib_SetDataMode(gDefaultDataMode_c);                          //Packet_Mode
         PhyPib_SetPowerStep(gDefaultOutputPower);                              //PA0 output power -1dBm for US. Commented for EU (13dBm default). Enabled for US
	 PhyPib_EnableAfcAutoclear(gDefaultAfcAutoClear_c);                     //FALSE
	 PhyPib_EnableAfcAuto(gDefaultAfcAuto_c);                               //FALSE
	 PhyPib_SetRxTimeout(gDefaultRxTimeout_c);                              //0x00
	 PhyPib_SetRxTimeoutThreshold(gDefaultRxTimeoutThresh_c);               //0x00
	 (void)PhyPib_EnableSensitivityBoost(gDefaultSensitivityBoost_c);       //Normal Sensitivity
	 PhyPib_SetPaRamp(gDefaultPaRamp_c);            
	 PhyPib_SetLnaGain(gDefaultLnaGain_c);
	 PhyPib_SetRssiThreshold(gDefaultRssiThresh_c);                         //0xFF
	 PhyPib_SetDccFreq(gDefaultDccFreq_c);                                  //DccFreq_7
	 PhyPib_SetRxFilterBw(gDefaultRxFilterBw_c);                            //83,300
	 PhyPib_SetDccFreqAfc(gDefaultDccFreqAfc_c);                            //DccFreq_5                          
         PhyPib_SetRxFilterBwAfc(gDefaultRxFilterBwAfc_c);                      //100,000
	 PhyPib_SetSyncWordValue(MKW01_Reg_SyncValue1, gDefaultSyncValue_c);
	 PhyPib_SetSyncWordValue(MKW01_Reg_SyncValue2, gDefaultSyncValue_c);
	 PhyPib_SetSyncWordValue(MKW01_Reg_SyncValue3, gDefaultSyncValue_c);
	 PhyPib_SetSyncWordValue(MKW01_Reg_SyncValue4, gDefaultSyncValue_c);
	 PhyPib_SetSyncWordValue(MKW01_Reg_SyncValue5, gDefaultSyncValue_c);
	 PhyPib_SetSyncWordValue(MKW01_Reg_SyncValue6, gDefaultSyncValue_c);
	 PhyPib_SetSyncWordValue(MKW01_Reg_SyncValue7, gDefaultSyncValue_c);
	 PhyPib_SetSyncWordValue(MKW01_Reg_SyncValue8, gDefaultSyncValue_c);
	 PhyPib_SetFifoThreshold(gDefaultFifoThresh_c);
         
         
        /*Settings for Wide Band Systems*/
        /*Refer to chapter Low Modulation Index Systems in the device datasheet for more details*/
         MKW01Drv_WriteRegister(MKW01_Reg_TestDagc, 0x30);                      //Improved margin, used when AfcLowBetaOn=0
         MKW01Drv_WriteRegister(MKW01_Reg_AfcCtrl, 0x00);                       //Standard AFC routine, for signals with modulation index (2*Fdev/BR) greater than 2.
         MKW01Drv_WriteRegister(MKW01_Reg_RssiThresh, 0xE4);                    //Default(-114 dBms).
         MKW01Drv_WriteRegister(MKW01_Reg_AfcFei, 0x18);                        //AfcAutoClearOn=1. AfcAutoOn=0 (AFC only performed after AfcStart is set. (No AFC for each packet, only once to compensate for the Tx frequency deviation)                   
   }

void Platform_Init(void) 
 {

 volatile uint8_t dummy;
 
        /* Init MCU specifics: SPI - and reset Radio */
        PlatformPortInit();                                                     
        LED_Init();                                                                                                   
       	MKW01Drv_SpiInit();                                                     // Init SPI for Radio driver
        
	/* Use external clock for testing */
	dummy = MKW01Drv_ReadRegister(0x59);
	dummy &= 0xEF;
	dummy |= 0x10;
	MKW01Drv_WriteRegister(0x59, dummy);   
         
 	(void)MKW01Drv_Reset();					                // Reset Radio  
        
        MKW01Drv_ConfigureCLKO(ClkOutFxOsc_Div16);                              //32Mhz/16= 2MHz CLKOUT
                   
        /*Initialize clock based on CLKOUT*/
        MCG_Setup();                                         
        MKW01Drv_WriteRegister(MKW01_Reg_PaRamp, 0x0A);           

        /* Timer module initialization */
        TMR_Init();  
        TMR_Task(); 
        
        Uart_ModuleInit();
	UartX_SetBaud(115200UL);
        
        /*Keyboard module initialization*/
  #if (gDeviceType_c == gTxNode_c)                                              //Only enable Keyboard for Tx node.
        Keyboard_Init(Keyboard_CallBack);
  #endif
 }
        

/************************************************************************************
*
* InitGlobalVariables
*
************************************************************************************/

void InitGlobalVariables(void)
{

  /* Initialize Global Variables */ 
   uint8_t payloadFill = 0;
   uint8_t payloadIndex = 27;
   uint16_t dummyWrite = 0;

   bRxDone = FALSE;
   bTxDone = FALSE;
   bTimeout = FALSE;
   bDelayTimeout = FALSE;
   bRxAborted = FALSE; 
    
  delayTimer = TMR_AllocateTimer();

    for(dummyWrite = 0; dummyWrite < 251; dummyWrite++)                    
   {
           gau8TxDataBuffer[dummyWrite] = 0x0E;
           gau8RxDataBuffer[dummyWrite] = 0x0E;
   }
   
   /* Initialization of SMAC variables */
   gAppRxPacket = (rxPacket_t *)gau8RxDataBuffer;
   gAppRxPacket->u8MaxDataLength = gMaxSmacSDULength_c;
   
   gAppTxPacket = (txPacket_t *)gau8TxDataBuffer;
   gAppTxPacket->smacPdu.u8DestAddress = gBroadcastAddress_c;
   gAppTxPacket->u8DataLength = gSimpleRangePayloadLength_c;
   FLib_MemCpy(gAppTxPacket->smacPdu.smacPdu, "Freescale Simple Range Demo", gSimpleRangePayloadLength_c);
   //FLib_MemCpy(gAppTxPacket->smacPdu.smacPdu, "Hi there", gSimpleRangePayloadLength_c);
   payloadFill = gSimpleRangePayloadLength_c - gSimpleRangeDefaultPayloadLength_c;
   while(payloadFill--)                                                         // If payload set by the user bigger than default 
   {
           gAppTxPacket->smacPdu.smacPdu[payloadIndex++] = payloadFill;
   }      
}

/************************************************************************************
*
* InitSmac
*
************************************************************************************/

void InitSmac(void)
{
            
   smacStandalone = FALSE;
   MLMERadioInit();  
   MLMESetPreambleLength(gDefaultPreambleLength_c);
   MLMESetSyncWordSize(gDefaultSyncWordSize_c);
   //MLMESetSyncWordValue(u8MySyncWord);                                        //Set in SetDefaultValues()
   //MLMEPAOutputAdjust(gMaxOutputPower_c);                                     //Set in SetDefaultValues()
   
#ifdef MKW01_NA
   MLMESetFreqBand(g902__928MHz_c, gRFMode0_c);                                 //Americas configuration (915 MHz)
#endif
#ifdef MKW01_EU
   MLMESetFreqBand(g863__870MHz_c, gRFMode0_c);                                 //Europe configuration (868.3MHz MHz). Changed in SetDefaultValues() to 868.525 MHz
#endif

}

 void PlatformPortInit(void)
 {
    /* Enable all of the port clocks. These have to be enabled to configure
     * pin muxing options, so most code will need all of these on anyway.
     */
    SIM_SCGC5 |= (SIM_SCGC5_PORTA_MASK
                  | SIM_SCGC5_PORTB_MASK
                  | SIM_SCGC5_PORTC_MASK
                  | SIM_SCGC5_PORTD_MASK
                  | SIM_SCGC5_PORTE_MASK );

    /*Need to set pin as input*/
    /* Port A, Pin 18 <- EXTAL */ 
    /* pins direction -> INPUT (PDD&=~0x20000) */
    PORTA_PCR18 = PORT_PCR_MUX(1);
    GPIOA_PDDR &= (uint32_t)~(uint32_t)(gMKW01ClkOutMask_c);

    /*Need to set IRQ pins as input*/ 
    /*For Radio pins DIO0, DIO1, DIO4*/
    MKW01xDrv_IRQ_PortConfig();
    GPIOC_PDDR &= (uint32_t)~(uint32_t)(1<<3);      //PTC3<-DIO0                
    GPIOC_PDDR &= (uint32_t)~(uint32_t)(1<<4);      //PTC4<-DIO1                
    GPIOD_PDDR &= (uint32_t)~(uint32_t)(1<<4);      //PTD4<-DIO4         
        
 }


/************************************************************************************
* User's Callbacks
************************************************************************************/

/* Place your callbacks here */
void UartTxCallBack(uint8_t *pBuf) {
 	(void) pBuf;
 }
/************************************************************************************
*
* TimerDelayCallback                                                            
*
************************************************************************************/
  void TimerDelayCallback(tmrTimerID_t timerId)
  {
	  bDelayTimeout = TRUE;
  }

/************************************************************************************
*
* Keyboard_CallBack                                                                    
*
************************************************************************************/
  void Keyboard_CallBack(uint8_t Keyboard_PortID)
  {
	  if(Keyboard_PortID == 3)                                              //SW3
	  {                                  
                  Led4Off(); /* turn OFF LED 4 */
		  MLMEEnablePABoost(gDisablePA_Boost_c);
	  }
	  else if(Keyboard_PortID == 4)                                         //SW4
	  {                                  
                  Led4On(); /* turn ON LED 4 */
		  MLMEEnablePABoost(gEnablePABoth_Boost_c);
		  
	  }
  }

/************************************************************************************
* SMAC Callbacks
************************************************************************************/


/************************************************************************************
*
* MCPSDataConfirm
*
************************************************************************************/

void MCPSDataConfirm(txStatus_t TransmissionResult)
{  
  bTxDone = TRUE;
}

/************************************************************************************
*
* MCPSDataIndication
*
************************************************************************************/

void MCPSDataIndication(rxPacket_t *gsRxPacket)
{  
   if(rxSuccessStatus_c == gsRxPacket->rxStatus)
   {
    bRxDone = TRUE;	    
   }
   else if (rxTimeOutStatus_c == gsRxPacket->rxStatus)
   {
    bTimeout = TRUE;
   }
   
   else if (rxAbortedStatus_c == gsRxPacket->rxStatus)
   {
    bRxAborted = TRUE;
   }
   
}

/* Table to calculate the FXTH CRC */
uint16_t cu16CCITTTable[0x100] = 
{
  0x0000u, 0x1021u, 0x2042u, 0x3063u, 0x4084u, 0x50a5u, 0x60c6u, 0x70e7u, /* 000 - 007 */
  0x8108u, 0x9129u, 0xa14au, 0xb16bu, 0xc18cu, 0xd1adu, 0xe1ceu, 0xf1efu, /* 008 - 015 */
  0x1231u, 0x0210u, 0x3273u, 0x2252u, 0x52b5u, 0x4294u, 0x72f7u, 0x62d6u, /* 016 - 023 */
  0x9339u, 0x8318u, 0xb37bu, 0xa35au, 0xd3bdu, 0xc39cu, 0xf3ffu, 0xe3deu, /* 024 - 031 */
  0x2462u, 0x3443u, 0x0420u, 0x1401u, 0x64e6u, 0x74c7u, 0x44a4u, 0x5485u, /* 032 - 039 */
  0xa56au, 0xb54bu, 0x8528u, 0x9509u, 0xe5eeu, 0xf5cfu, 0xc5acu, 0xd58du, /* 040 - 047 */
  0x3653u, 0x2672u, 0x1611u, 0x0630u, 0x76d7u, 0x66f6u, 0x5695u, 0x46b4u, /* 048 - 055 */
  0xb75bu, 0xa77au, 0x9719u, 0x8738u, 0xf7dfu, 0xe7feu, 0xd79du, 0xc7bcu, /* 056 - 063 */
  0x48c4u, 0x58e5u, 0x6886u, 0x78a7u, 0x0840u, 0x1861u, 0x2802u, 0x3823u, /* 064 - 071 */
  0xc9ccu, 0xd9edu, 0xe98eu, 0xf9afu, 0x8948u, 0x9969u, 0xa90au, 0xb92bu, /* 072 - 079 */
  0x5af5u, 0x4ad4u, 0x7ab7u, 0x6a96u, 0x1a71u, 0x0a50u, 0x3a33u, 0x2a12u, /* 080 - 087 */
  0xdbfdu, 0xcbdcu, 0xfbbfu, 0xeb9eu, 0x9b79u, 0x8b58u, 0xbb3bu, 0xab1au, /* 088 - 095 */
  0x6ca6u, 0x7c87u, 0x4ce4u, 0x5cc5u, 0x2c22u, 0x3c03u, 0x0c60u, 0x1c41u, /* 096 - 103 */
  0xedaeu, 0xfd8fu, 0xcdecu, 0xddcdu, 0xad2au, 0xbd0bu, 0x8d68u, 0x9d49u, /* 104 - 111 */
  0x7e97u, 0x6eb6u, 0x5ed5u, 0x4ef4u, 0x3e13u, 0x2e32u, 0x1e51u, 0x0e70u, /* 112 - 119 */
  0xff9fu, 0xefbeu, 0xdfddu, 0xcffcu, 0xbf1bu, 0xaf3au, 0x9f59u, 0x8f78u, /* 120 - 127 */
  0x9188u, 0x81a9u, 0xb1cau, 0xa1ebu, 0xd10cu, 0xc12du, 0xf14eu, 0xe16fu, /* 128 - 135 */
  0x1080u, 0x00a1u, 0x30c2u, 0x20e3u, 0x5004u, 0x4025u, 0x7046u, 0x6067u, /* 136 - 143 */
  0x83b9u, 0x9398u, 0xa3fbu, 0xb3dau, 0xc33du, 0xd31cu, 0xe37fu, 0xf35eu, /* 144 - 151 */
  0x02b1u, 0x1290u, 0x22f3u, 0x32d2u, 0x4235u, 0x5214u, 0x6277u, 0x7256u, /* 152 - 159 */
  0xb5eau, 0xa5cbu, 0x95a8u, 0x8589u, 0xf56eu, 0xe54fu, 0xd52cu, 0xc50du, /* 160 - 167 */
  0x34e2u, 0x24c3u, 0x14a0u, 0x0481u, 0x7466u, 0x6447u, 0x5424u, 0x4405u, /* 168 - 175 */
  0xa7dbu, 0xb7fau, 0x8799u, 0x97b8u, 0xe75fu, 0xf77eu, 0xc71du, 0xd73cu, /* 176 - 183 */
  0x26d3u, 0x36f2u, 0x0691u, 0x16b0u, 0x6657u, 0x7676u, 0x4615u, 0x5634u, /* 184 - 191 */
  0xd94cu, 0xc96du, 0xf90eu, 0xe92fu, 0x99c8u, 0x89e9u, 0xb98au, 0xa9abu, /* 192 - 199 */
  0x5844u, 0x4865u, 0x7806u, 0x6827u, 0x18c0u, 0x08e1u, 0x3882u, 0x28a3u, /* 200 - 207 */
  0xcb7du, 0xdb5cu, 0xeb3fu, 0xfb1eu, 0x8bf9u, 0x9bd8u, 0xabbbu, 0xbb9au, /* 208 - 215 */
  0x4a75u, 0x5a54u, 0x6a37u, 0x7a16u, 0x0af1u, 0x1ad0u, 0x2ab3u, 0x3a92u, /* 216 - 223 */
  0xfd2eu, 0xed0fu, 0xdd6cu, 0xcd4du, 0xbdaau, 0xad8bu, 0x9de8u, 0x8dc9u, /* 224 - 231 */
  0x7c26u, 0x6c07u, 0x5c64u, 0x4c45u, 0x3ca2u, 0x2c83u, 0x1ce0u, 0x0cc1u, /* 232 - 239 */
  0xef1fu, 0xff3eu, 0xcf5du, 0xdf7cu, 0xaf9bu, 0xbfbau, 0x8fd9u, 0x9ff8u, /* 240 - 247 */
  0x6e17u, 0x7e36u, 0x4e55u, 0x5e74u, 0x2e93u, 0x3eb2u, 0x0ed1u, 0x1ef0u  /* 248 - 255 */
};
/*
Input Parameters:
 UINT8 *u8Buffer: Buffer where data is located.
 UINT16 u16MByteSize: Size of the designated buffer (in bytes)
 UINT16 u16Remainder: Initial remainder.
 Returns:
 UINT16 u16NewCRC: New calculated CRC16.
*/
uint16_t Calculate_CRC_FXTH(uint8_t* pu8Buffer, uint16_t u16BufferSize, uint16_t u16Remainder)
{
  /* Loop while we have data in the buffer */
  for(;u16BufferSize > 0; u16BufferSize--)
  {
    /* XOR MSB of Remainder with the buffer to generate index                */
    /* XOR MSB of value stored in table with LSB of previous Remainder to    */
    /* generate new MSB for remainder; LSB is the LSB of the found value.    */
    u16Remainder = ((uint16_t)u16Remainder << (uint16_t)8) ^ cu16CCITTTable[0xFF &((uint8_t)(u16Remainder >> (uint16_t)8) ^ *pu8Buffer)];
      
    /* Increase pointer to the next byte in the stream */
    pu8Buffer++;
  }
  return(u16Remainder);
}

uint8_t FXTH_CRC_Verification(void){
  uint8_t Result;
  uint8_t PayloadLength;
  uint16_t Calculated_CRC;  
  uint16_t Received_CRC;
  PayloadLength = gau8RxDataBuffer[2];
  Calculated_CRC = 0;
  Received_CRC = (gau8RxDataBuffer[2+PayloadLength+1])&0x00FF;
  Received_CRC <<= 8;
  Received_CRC |= gau8RxDataBuffer[2+PayloadLength+2];
  
  gau8RxDataBuffer[2] += 2; /* This value has been modified at reception => re-add the offset to calculate the rigth CRC value */
  Calculated_CRC = Calculate_CRC_FXTH(&gau8RxDataBuffer[2], (uint16_t)(PayloadLength + 1), 0);
  if (Calculated_CRC == Received_CRC){
    Result = 0x00;
  }
  else {
    Result = 0x22; /* If FXTH CRC is not correct, return 0x22 */
  }
  
  return Result;
}

uint8_t MKW01_CRC_Verification(void){
  uint8_t Result;
  if ((irqFlags2&0x02) == 0x02){
    Result = 0x00;
  }
  else {
    Result = 0x33; /* If MKW01 CRC is not correct, return 0x33 */
  }
  
  return Result;
}

uint8_t Checksum_Verification (void)
{  
  uint8_t Result;
  uint8_t Received_Checksum;
  uint8_t Calculated_Checksum;
  uint8_t Payload_Length;
  int i;
  
  Calculated_Checksum = 0x00;
  gau8RxDataBuffer[2] += 2; /* This value has been modified at reception => re-add the offset to calculate the rigth checksum value */
  Payload_Length = gau8RxDataBuffer[2];
  
  for (i=2; i<(Payload_Length + 1); i++){
    Calculated_Checksum += gau8RxDataBuffer[i];
  }
  Calculated_Checksum = ((uint8_t)0xFFu) - Calculated_Checksum;
  Received_Checksum = gau8RxDataBuffer[Payload_Length+2];
  
  if (Calculated_Checksum == Received_Checksum){
    Result = 0x00;
  }
  else {
    Result = 0x11; /* If checksum is not correct, return 0x11 */
  }
  
  return Result;
}

uint8_t Frame_Verification(void)
{
  uint8_t Result;
  switch (gau8RxDataBuffer[19]) /* Contains Verification type */
  {
  case (0x00): Result = 0x44; /* No verification => result = 0x44 */
  break;
  
  case (0x01): Result = Checksum_Verification();
  break;
  
  case (0x02): Result = FXTH_CRC_Verification();
  break;
  
  case (0x03): Result = MKW01_CRC_Verification();
  break;
  
  default: Result = 0x00;
  }
  
 return Result;   
}

